#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include "../include/ebtables_u.h"
#include <linux/netfilter_bridge/ebt_tagstrip.h>

#define TAGSTRIP_TARGET '1'
static struct option opts[] =
{
	{ EBT_TAGSTRIP_TARGET "-target", required_argument, 0, TAGSTRIP_TARGET },
	{ 0 }
};

static void print_help()
{
	printf(
	EBT_TAGSTRIP_TARGET " target options:\n"
	" --" EBT_TAGSTRIP_TARGET "-target target     : ACCEPT, DROP, RETURN or CONTINUE\n");
}

static void init(struct ebt_entry_target *target)
{
	struct ebt_tagstrip_info *info = (struct ebt_tagstrip_info *)target->data;

	info->target = EBT_CONTINUE;
}

#define OPT_TAGSTRIP_TARGET   0x01
static int parse(int c, char **argv, int argc,
   const struct ebt_u_entry *entry, unsigned int *flags,
   struct ebt_entry_target **target)
{
	struct ebt_tagstrip_info *info = (struct ebt_tagstrip_info *)(*target)->data;

	switch(c) {
	case TAGSTRIP_TARGET:
		{ int tmp;
		/* check for double usage of this option */
		ebt_check_option2(flags, OPT_TAGSTRIP_TARGET);
		if (FILL_TARGET(optarg, tmp))
			ebt_print_error2("Illegal --" EBT_TAGSTRIP_TARGET "-target target");
		/* the 4 lsb are left to designate the target */
		info->target = (info->target & ~EBT_VERDICT_BITS) | (tmp & EBT_VERDICT_BITS);
		}
		break;
	default:
		return 0;
	}
	return 1;
}

static void final_check(const struct ebt_u_entry *entry,
   const struct ebt_entry_target *target, const char *name,
   unsigned int hookmask, unsigned int time)
{
	struct ebt_tagstrip_info *info =
	   (struct ebt_tagstrip_info *)target->data;

	if (entry->ethproto != ETH_P_8021Q) {
		ebt_print_error("For " EBT_TAGSTRIP_TARGET " the protocol must be specified as 802.1Q");
	} else if (BASE_CHAIN && (info->target | ~EBT_VERDICT_BITS) == EBT_RETURN) {
		ebt_print_error("--" EBT_TAGSTRIP_TARGET "-target RETURN not allowed on base chain");
	} else {
		CLEAR_BASE_CHAIN_BIT;
		if ((hookmask & ~(1 << NF_BR_LOCAL_IN)) || strcmp(name, "filter"))
			ebt_print_error("Wrong chain for " EBT_TAGSTRIP_TARGET);
	}
}

static void print(const struct ebt_u_entry *entry,
   const struct ebt_entry_target *target)
{
	struct ebt_tagstrip_info *info =
	   (struct ebt_tagstrip_info *)target->data;

	if (info->target == EBT_CONTINUE)
		return;
	printf(" --" EBT_TAGSTRIP_TARGET "-target %s", TARGET_NAME(info->target));
}

static int compare(const struct ebt_entry_target *t1,
   const struct ebt_entry_target *t2)
{
	struct ebt_tagstrip_info *info1 =
	   (struct ebt_tagstrip_info *)t1->data;
	struct ebt_tagstrip_info *info2 =
	   (struct ebt_tagstrip_info *)t2->data;

	return info1->target == info2->target;
}

static struct ebt_u_target tagstrip_target =
{
	.name		= EBT_TAGSTRIP_TARGET,
	.size		= sizeof(struct ebt_tagstrip_info),
	.help		= print_help,
	.init		= init,
	.parse		= parse,
	.final_check	= final_check,
	.print		= print,
	.compare	= compare,
	.extra_ops	= opts,
};

void _init(void)
{
	ebt_register_target(&tagstrip_target);
}
