#include <linux/irq.h>
#include <linux/cdev.h>
#include <linux/module.h>
#include <linux/kdev_t.h>
#include <linux/poll.h>
#include <linux/moduleparam.h>
#include <linux/types.h>
#include <linux/timer.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/fs.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/init.h>
#include <linux/jiffies.h>
#include <asm/uaccess.h>
#include <linux/autoconf.h>
#include <linux/interrupt.h>
#ifdef IXP_GPIO
#include <asm/hardware.h>
#endif

#define MODULE_NAME  "dvlheartbeat"
MODULE_LICENSE("GPL");

static void start_timer(void);

static struct timer_list wdt;
#ifdef IXP_GPIO
static int gpiostate = IXP4XX_GPIO_LOW;
#endif

#define TIMER_MARGIN (5*60)            /* Default is 5 minutes */
static int soft_margin = TIMER_MARGIN; /* in seconds */
static unsigned long driver_open, orphan_timer;
static int counter = TIMER_MARGIN;

static unsigned int dvlheartbeat_major = 244; // 0=use automatic majornumber, not static
                                              // can be read from /proc/devices
static unsigned int dvlheartbeat_minor = 0;   // use automatic minornumber

struct dvlheartbeat_dev {
    struct semaphore sem;
    struct cdev cdev;         /* character device */
};

static dev_t dev_id;
static struct dvlheartbeat_dev global_dev;


/*
 *  timer operations
 */

static void wdt_handler(unsigned long unused_arg)
{
#ifdef IXP_GPIO
    gpiostate = (IXP4XX_GPIO_LOW==gpiostate ) ? IXP4XX_GPIO_HIGH : IXP4XX_GPIO_LOW;  
    gpio_line_set(DVLHOST_WDI, gpiostate);
#endif
    start_timer();
}

static void start_timer(void)
{
    init_timer(&wdt);
    wdt.data = 0;
    wdt.function = wdt_handler;
    wdt.expires = jiffies + HZ; //1sec
    counter = counter - 1;

    if(counter <= 0)
    {
        printk(KERN_CRIT "Watchdog initiates system reboot.\n");
        emergency_restart();
        printk(KERN_CRIT "Reboot didn't ?????\n");
    }
    else
    {
        add_timer(&wdt);
    }
}

/*
 *  Watchdog operations
 */

static int wdt_keepalive()
{
    counter = soft_margin;
    return 0;
}

static int wdt_stop(void)
{
    // todo 
    return 0;
}

static int wdt_set_heartbeat(int t)
{
    if ((t < 0x0001) || (t > 0xFFFF))
        return -EINVAL;

    soft_margin = t;
    return 0;
}


/*
 *  /dev/MODULE_NAME handling
 */

static int wdt_open(struct inode *inode, struct file *file)
{
    int rv;
    if (test_and_set_bit(0, &driver_open))
        return -EBUSY;
    if (!test_and_clear_bit(0, &orphan_timer))
        __module_get(THIS_MODULE);

    rv = nonseekable_open(inode, file);
    return rv;
}

static int wdt_release(struct inode *unused_inode, struct file *unused_file)
{
    int rv = 0;
    set_bit(0, &orphan_timer);
    clear_bit(0, &driver_open);
    return rv;
}


static ssize_t wdt_write(struct file *file, const char __user *data, size_t len, loff_t *ppos)
{
    /*
     *  Refresh the timer.
     */
    if(len) {
        wdt_keepalive();
    }
    return len;
}

/*
 * Called if read() is called
 */
static ssize_t wdt_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos) {
    ssize_t retval = 0;

	char *s = (char *) kzalloc((16), GFP_KERNEL);
	if (s == NULL)
	{
		printk(KERN_ERR "unable to allocate memory\n");
	}
    else
    {
		char line[16];
		snprintf(line, sizeof(line), "%d\n", counter);
		strcat(s, line);

        if (copy_to_user(buf, s, strlen(s)))
            retval = -EFAULT;
        else
            retval = (ssize_t) strlen(s);
        
        kfree(s);
    }

    return retval;
}


static int wdt_ioctl(struct inode *inode, struct file *file,
                     unsigned int cmd, unsigned long arg)
{
    void __user *argp = (void __user *)arg;
    int __user *p = argp;
    int new_margin;
    static struct watchdog_info ident = {
        .options =  WDIOF_SETTIMEOUT |
        WDIOF_KEEPALIVEPING |
        WDIOF_MAGICCLOSE,
        .firmware_version = 0,
        .identity = MODULE_NAME,
    };
    switch (cmd) {
        default:
            return -ENOTTY;
        case WDIOC_GETSUPPORT:
            return copy_to_user(argp, &ident,
                                sizeof(ident)) ? -EFAULT : 0;
        case WDIOC_GETSTATUS:
        case WDIOC_GETBOOTSTATUS:
            return put_user(0, p);
        case WDIOC_KEEPALIVE:
        {
            int rv;
            rv = wdt_keepalive();
            return rv;
        }
        case WDIOC_SETTIMEOUT:
            if (get_user(new_margin, p))
                return -EFAULT;
            if (wdt_set_heartbeat(new_margin))
                return -EINVAL;
            wdt_keepalive();
            /* Fall */
        case WDIOC_GETTIMEOUT:
            return put_user(soft_margin, p);
    }
}

/*
 *  Notifier for system down
 */

static int wdt_notify_sys(struct notifier_block *this, unsigned long code,
                          void *unused)
{
    if(code==SYS_DOWN || code==SYS_HALT) {
        /* Turn the WDT off */
        wdt_stop();
    }
    return NOTIFY_DONE;
}


/*
 *  Kernel Interfaces
 */

static const struct file_operations wdt_fops = {
    .owner      = THIS_MODULE,
    .llseek     = no_llseek,
    .write      = wdt_write,
    .read       = wdt_read,
    .ioctl      = wdt_ioctl,
    .open       = wdt_open,
    .release    = wdt_release,
};

static struct notifier_block wdt_notifier = {
    .notifier_call  = wdt_notify_sys,
};


/*
 * Setup character devices
 */
static void dvlheartbeat_setup_cdev(struct dvlheartbeat_dev *dev, unsigned int index) {
    int err;
    unsigned int devno = MKDEV(dvlheartbeat_major, dvlheartbeat_minor + index);

    cdev_init(&dev->cdev, &wdt_fops);
    dev->cdev.owner = THIS_MODULE;
    dev->cdev.ops = &wdt_fops;
    err = cdev_add(&dev->cdev, devno, 1);
    init_MUTEX(&dev->sem);

    if (err)
        printk(KERN_DEBUG "Error %d adding %s%d",err,MODULE_NAME,index);
}

/*
 *  module
 */

static int __init init_mod(void)
{
    int ret;    

    dev_id = MKDEV(dvlheartbeat_major, dvlheartbeat_minor);
    ret = register_chrdev_region(dev_id,1,MODULE_NAME);
    if (ret < 0) {
        printk(KERN_WARNING "%s, can't get major %d\n",MODULE_NAME, dvlheartbeat_major);
        return ret;
    }
    dvlheartbeat_setup_cdev(&global_dev, 0);

    ret = register_reboot_notifier(&wdt_notifier);
    if (ret) {
        printk (KERN_ERR "cannot register reboot notifier (err=%d)\n",
                ret);
        cdev_del(&global_dev.cdev);
        unregister_chrdev_region(dev_id,1);
        return ret;
    }

#ifdef IXP_GPIO
    gpio_line_config(DVLHOST_WD_LED_ENABLE, IXP4XX_GPIO_OUT);
    gpio_line_set(DVLHOST_WD_LED_ENABLE, IXP4XX_GPIO_LOW);
    gpio_line_config(DVLHOST_WDI, IXP4XX_GPIO_OUT);
    gpio_line_set(DVLHOST_WDI, gpiostate);
#endif
    start_timer();

    printk(KERN_INFO "Module %s initialized\n", MODULE_NAME);   
    return 0;
}

static void __exit exit_mod(void)
{
    unregister_reboot_notifier(&wdt_notifier);
    cdev_del(&global_dev.cdev);
    unregister_chrdev_region(dev_id,1);

    del_timer_sync(&wdt); 
#ifdef IXP_GPIO
    gpio_line_config(DVLHOST_WDI, IXP4XX_GPIO_IN);
    gpio_line_config(DVLHOST_WD_LED_ENABLE, IXP4XX_GPIO_IN);
#endif
    printk(KERN_INFO "Module %s removed\n", MODULE_NAME);
}

MODULE_AUTHOR("Christian Taedcke");
MODULE_DESCRIPTION("devolo heartbeat module");

module_init(init_mod);
module_exit(exit_mod);

//EXPORT_NO_SYMBOLS;
