
/*
 * This code is just a copy of ebtables/ .. /examples/ulog/test_ulog.c
 * which is adapted to our needs.
 */

#include <asm/types.h>
#include <sys/socket.h>
#include <linux/netlink.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <netdb.h>
#include <errno.h>
#include <netinet/if_ether.h>
#include <netinet/ether.h>
#include <netinet/ip.h>
#include "include/ebtables_u.h"
#include <linux/netfilter_bridge/ebt_ulog.h>


#define ARP_HLEN 8
#define ARP_DLEN 20

/*
 * ARP data (behind ARP header) for IP over Ethernet
 */
struct arpdata
{
    unsigned char       ar_sha[ETH_ALEN];   /* sender hardware address */
    unsigned char       ar_sip[4];          /* sender IP address       */
    unsigned char       ar_tha[ETH_ALEN];   /* target hardware address */
    unsigned char       ar_tip[4];          /* target IP address       */
};


static struct sockaddr_nl sa_local =
{
    .nl_family = AF_NETLINK,
    .nl_groups = 1,
};

static struct sockaddr_nl sa_kernel =
{
    .nl_family = AF_NETLINK,
    .nl_pid = 0,
    .nl_groups = 1,
};

static const char *hookstr[NF_BR_NUMHOOKS] =
{
        [NF_BR_POST_ROUTING] "POSTROUTING",
         [NF_BR_PRE_ROUTING] "PREROUTING",
           [NF_BR_LOCAL_OUT] "OUTPUT",
            [NF_BR_LOCAL_IN] "INPUT",
            [NF_BR_BROUTING] "BROUTING",
             [NF_BR_FORWARD] "FORWARD"
};

#define DEBUG_QUEUE 0
#define BUFLEN 65536
static char buf[BUFLEN];
static int sfd;

#define Dprintf(fmt, ...) do  \
if (debug == 1)               \
printf(fmt , ##__VA_ARGS__);  \
while(0)


/* Get the next ebt_ulog packet, talk to the kernel if necessary */
ebt_ulog_packet_msg_t *ulog_get_packet()
{
    static struct nlmsghdr *nlh = NULL;
    static int len, remain_len;
    static int pkts_per_msg = 0;
    ebt_ulog_packet_msg_t *msg;
    socklen_t addrlen = sizeof(sa_kernel);

    if (!nlh) {
recv_new:
        if (pkts_per_msg && DEBUG_QUEUE)
            printf("PACKETS IN LAST MSG: %d\n", pkts_per_msg);
        pkts_per_msg = 0;
        len = recvfrom(sfd, buf, BUFLEN, 0,
                       (struct sockaddr *)&sa_kernel, &addrlen);
        if (addrlen != sizeof(sa_kernel)) {
            printf("addrlen %u != %u\n", addrlen,
                   (uint32_t)sizeof(sa_kernel));
            exit(-1);
        }
        if (len == -1) {
            perror("recvmsg");
            if (errno == EINTR)
                goto recv_new;
            exit(-1);
        }
        nlh = (struct nlmsghdr *)buf;
        if (nlh->nlmsg_flags & MSG_TRUNC || len > BUFLEN) {
            printf("Packet truncated");
            exit(-1);
        }
        if (!NLMSG_OK(nlh, BUFLEN)) {
            perror("Netlink message parse error\n");
            return NULL;
        }
    }

    msg = NLMSG_DATA(nlh);

    remain_len = (len - ((char *)nlh - buf));
    if (nlh->nlmsg_flags & NLM_F_MULTI && nlh->nlmsg_type != NLMSG_DONE)
        nlh = NLMSG_NEXT(nlh, remain_len);
    else
        nlh = NULL;

    pkts_per_msg++;
    return msg;
}


#define USAGE    "ulogd\n" \
        "Usage:  ulogd <nlgroup> <packettype> {options}\n" \
        "\n" \
        "ulogd is a userspace logging daemon which is used in conjunction with ebtables.\n" \
        "Add the appropriate ebtables ulog rule, e.g. (<nlgroup>: [1..32]):\n" \
        " ebtables -A FORWARD --in-interface ath0 -p 0x0806 --ulog-prefix ARP --ulog-nlgroup <nlgroup> .\n" \
        "Start ulogd daemon, e.g.:\n" \
        " ulogd <nlgroup> arp -d .\n" \
        "\n" \
        "Netlink group number:\n" \
        "  1 .. 32\n" \
        "\n" \
        "Packettype:\n" \
        "  arp      evaluate ARP packets and get pairs of 'macAddress,ipAddress'\n" \
        "\n" \
        "Options:\n" \
        "  -d       generate debug output\n" \
        "  -h       help\n" \
        "\n"


int main(int argc, char **argv)
{
    int i, curr_len, pktcnt = 0;
    int rcvbufsize = BUFLEN;
    ebt_ulog_packet_msg_t *msg;
    struct ethhdr *ehdr;
    struct protoent *prototype;
    struct iphdr *iph;
    char *ctmp;
    struct arpdata *adata;
    int debug = 0;
    int opt;

    if (argc < 3) {
        printf( USAGE );
        return 0;
    }

    i = strtoul(argv[1], &ctmp, 10);
    if (*ctmp != '\0' || i < 1 || i > 32) {
        printf( USAGE );
        return 0;
    }
    sa_local.nl_groups = sa_kernel.nl_groups = 1 << (i - 1);

    if ( strcmp( argv[2], "arp" ) != 0 ) {
        printf( USAGE );
        return 0;
    }

    while ((opt = getopt (argc, argv, "dh")) != -1) {
        switch(opt)
        {
            case 'd':
                debug = 1;
                break;
            case 'h':
            case '?':
               printf( USAGE );
               return 0;
        }
    }

    sa_local.nl_pid = getpid();
    sfd = socket(PF_NETLINK, SOCK_RAW, NETLINK_NFLOG);
    if (!sfd) {
        perror("socket");
        exit(-1);
    }

    if (bind(sfd, (struct sockaddr *)(&sa_local), sizeof(sa_local)) == -1) {
        perror("bind");
        exit(-1);
    }
    i = setsockopt(sfd, SOL_SOCKET, SO_RCVBUF, &rcvbufsize,
                   sizeof(rcvbufsize));

    while (1) {
        if (!(msg = ulog_get_packet()))
            continue;

        if (msg->version != EBT_ULOG_VERSION) {
            printf("WRONG VERSION: %d INSTEAD OF %d\n",
                   msg->version, EBT_ULOG_VERSION);
            continue;
        }

        Dprintf("PACKET NR: %d\n", ++pktcnt);
        iph = NULL;
        curr_len = ETH_HLEN;

        Dprintf("INDEV=%s\nOUTDEV=%s\nPHYSINDEV=%s\nPHYSOUTDEV=%s\n"
               "PREFIX='%s'", msg->indev, msg->outdev, msg->physindev,
               msg->physoutdev, msg->prefix);

        Dprintf("\nMARK=%lu\nHOOK=%s\n", msg->mark, hookstr[msg->hook]);

        if (msg->data_len < curr_len) {
            Dprintf("====>Packet smaller than Ethernet header length<====\n");
            goto letscontinue;
        }

        Dprintf("::::::::ETHERNET:HEADER::::::::\n");

        ehdr = (struct ethhdr *)msg->data;
        Dprintf("MAC SRC=%s\n", ether_ntoa((const struct ether_addr *)
               ehdr->h_source));
        Dprintf("MAC DST=%s\nETHERNET PROTOCOL=", ether_ntoa(
               (const struct ether_addr *)ehdr->h_dest));
        Dprintf("0x%x\n", ntohs(ehdr->h_proto));

        if (ehdr->h_proto == htons(ETH_P_ARP)) {
            struct arphdr *ahdr = (struct arphdr *)
                                      (((char *)ehdr) + curr_len);
            Dprintf("::::::::::ARP:HEADER::::::::::\n");
            curr_len += ARP_HLEN;

            if (msg->data_len < curr_len) {
                Dprintf("====>Packet only contains partial "
                       "ARP header<====\n");
                goto letscontinue;
            }

            Dprintf("  ahdr->ar_hrd=0x%04x\n", ntohs(ahdr->ar_hrd));
            Dprintf("  ahdr->ar_pro=0x%04x\n", ntohs(ahdr->ar_pro));
            Dprintf("  ahdr->ar_hln=0x%02x\n", ntohs(ahdr->ar_hln));
            Dprintf("  ahdr->ar_pln=0x%02x\n", ntohs(ahdr->ar_pln));
            Dprintf("  ahdr->ar_op=0x%04x\n", ntohs(ahdr->ar_op));

            if (ahdr->ar_hrd != htons(ARPHRD_ETHER))
            {
                Dprintf("Packet ignored: ARP Hardware type (0x%04x) != ARPHRD_ETHER (0x%04x)\n", ntohs(ahdr->ar_hrd), ntohs(ARPHRD_ETHER));
                goto letscontinue;
            }

            if (ahdr->ar_pro != htons(ETH_P_IP))
            {
                Dprintf("Packet ignored: ARP Protocol type (0x%04x) != ETH_P_IP (0x%04x)\n", ntohs(ahdr->ar_pro), ntohs(ETH_P_IP));
                goto letscontinue;
            }

            if (ahdr->ar_hln != htons(ETH_ALEN))
            {
                Dprintf("Packet ignored: Hardware size (0x%02x) != ETH_ALEN (0x%02x)\n", ntohs(ahdr->ar_hln), ntohs(ETH_ALEN));
                goto letscontinue;
            }

            if (ahdr->ar_pln != htons(4))
            {
                Dprintf("Packet ignored: Hardware size (0x%02x) != 4 (0x%02x)\n", ntohs(ahdr->ar_pln), 4);
                goto letscontinue;
            }

            if (ahdr->ar_op != htons(ARPOP_REQUEST))
            {
                Dprintf("Packet ignored: Opcode (0x%02x) != ARPOP_REQUEST (0x%02x)\n", ntohs(ahdr->ar_op), ntohs(ARPOP_REQUEST));
                goto letscontinue;
            }

            Dprintf("::::::::::ARP:HEADER::::::::::\n");
            adata = (struct arpdata *) (((char *)ahdr) + ARP_HLEN);
            curr_len += ARP_DLEN;

            if (msg->data_len < curr_len) {
                Dprintf("====>Packet only contains partial ARP data<====\n");
                goto letscontinue;
            }

            Dprintf("Sender MAC Adress=");
            for (i = 0; i < 6; i++)
                Dprintf("%02x%s", ((unsigned char *)&adata->ar_sha)[i],
                   (i == 5) ? "" : ":");
            Dprintf("\n");

            Dprintf("Sender IP Adress=");
            for (i = 0; i < 4; i++)
                Dprintf("%d%s", ((unsigned char *)&adata->ar_sip)[i],
                   (i == 3) ? "" : ".");
            Dprintf("\n");

            Dprintf("Target MAC Adress=");
            for (i = 0; i < 6; i++)
                Dprintf("%02x%s", ((unsigned char *)&adata->ar_tha)[i],
                   (i == 5) ? "" : ":");
            Dprintf("\n");

            Dprintf("Target IP address=");
            for (i = 0; i < 4; i++)
                Dprintf("%d%s", ((unsigned char *)&adata->ar_tip)[i],
                   (i == 3) ? "" : ".");
            Dprintf("\n");


            /* if Sender IP Adress is != 0.0.0.0: print pair <macAddress>,<ipAddress> */
            if ( (adata->ar_sip[0] != 0) ||
                 (adata->ar_sip[1] != 0) ||
                 (adata->ar_sip[2] != 0) ||
                 (adata->ar_sip[3] != 0) )
            {
                /* Sender MAC Adress */
                for (i = 0; i < 6; i++)
                    printf("%02x%s", ((unsigned char *)&adata->ar_sha)[i],
                       (i == 5) ? "" : ":");
                printf(",");

                /* Sender IP Adress */
                for (i = 0; i < 4; i++)
                    printf("%d%s", ((unsigned char *)&adata->ar_sip)[i],
                       (i == 3) ? "" : ".");
                printf("\n");
            }

        }

letscontinue:
        Dprintf("===>Total Packet length: %ld, of which we examined "
               "%d bytes\n", msg->data_len, curr_len);
        Dprintf("###############################\n"
               "######END#OF##PACKET#DUMP######\n"
               "###############################\n");
    }

    return 0;
}
